<?php

namespace GP_Entry_Blocks\Blocks;

use GFAPI;
use GP_Entry_Blocks\GF_Queryer;
use function GP_Entry_Blocks\replace_merge_tags;

/**
 * Entry Loop block which takes its inner blocks and loops over them for the queried entries.
 *
 * Inner blocks are parsed for merge tags for each entry. Additionally, each entry will be placed in its own container.
 *
 * @since 1.0
 */
class Entries_Loop extends Block {
	public function __construct( $path ) {
		parent::__construct( $path );

		add_filter( 'gpeb_loop_entry_content', array( $this, 'replace_loop_item_template_merge_tags' ), 5, 3 );

		if ( isset( $GLOBALS['wp_embed'] ) ) {
			add_filter( 'gpeb_loop_entry_content', array( $GLOBALS['wp_embed'], 'autoembed' ), 8 );
		}
	}

	public function replace_loop_item_template_merge_tags( $template, $entry_form, $entry ) {
		return replace_merge_tags( $template, $entry_form, $entry );
	}

	public function render( $attributes, $content, $block ) {
		$form_id = $block->context['gp-entry-blocks/formId'];
		$form    = GFAPI::get_form( $form_id );

		$queryer = GF_Queryer::attach( $block->context );
		$entries = $queryer->entries;

		if ( ! $queryer->is_entries() ) {
			return null;
		}

		if ( ! $entries || ! count( $entries ) || is_wp_error( $entries ) ) {
			$no_entries_found_message = rgar( $attributes, 'noEntriesFoundMessage', esc_html__( 'No entries found.', 'gp-entry-blocks' ) );
			return '<p>' . replace_merge_tags( $no_entries_found_message, $form ) . '</p>';
		}

		$opening_tag_pattern = '/^(<([a-zA-Z]+)\s+.*?class="(.*?)".*?>)/m';
		preg_match( $opening_tag_pattern, $content, $block_opening_tag_matches );
		list( , $block_full_opening_tag, $block_tag, $block_class ) = $block_opening_tag_matches;

		$loop_item_template = $content;
		$loop_item_template = str_replace( $block_full_opening_tag, '', $loop_item_template );
		$loop_item_template = preg_replace( "/<\/{$block_tag}>$/", '', $loop_item_template );

		$content = $block_full_opening_tag;

		foreach ( $entries as $entry ) {
			/**
			 * Filters the entry object about to be rendered in the current block.
			 *
			 * @param array $entry The current entry being rendered.
			 * @param array $form The form for the current entry.
			 * @param array $block The block object for the current block.
			 */
			$entry      = apply_filters( 'gpeb_entry', $entry, $form, $block );
			/**
			 * Filters the form object for the current entry in the current block.
			 *
			 * @param array $form The form for the current entry.
			 * @param array $entry The current entry being rendered.
			 * @param array $block The block object for the current block.
			 */
			$entry_form = apply_filters( 'gpeb_entry_form', $form, $entry, $block );

			$content .= '<div class="gp-entry-blocks-entry">';

			/**
			 * Filters each entry's content in the Entries Loop block.
			 *
			 * @param string $loop_item_template String containing the inner blocks for an individual entry that
			 *                                   will be used as the template for each entry.
			 * @param array $entry_form The form for the current entry.
			 * @param array $entry The current entry being rendered in the loop.
			 *
			 * @since 1.0-alpha-1.13
			 */
			$content .= apply_filters( 'gpeb_loop_entry_content', $loop_item_template, $entry_form, $entry );

			$content .= '</div>';
		}

		$content .= "</{$block_tag}>";

		return $content;
	}
}
