import memoize from 'fast-memoize';
import { __ } from '@wordpress/i18n';
import React from 'react';
import { FilterProperties, FilterPropertyGroups } from '../FiltersControl';

const getPropertiesByGroup = memoize(
	(
		properties: FilterProperties,
		group: string | undefined
	): FilterProperties => {
		return Object.keys( properties )
			.filter( ( propertyId ) => {
				const property = properties[ propertyId ];

				return property?.group === group;
			} )
			.reduce( ( res, propertyId ) => {
				res[ propertyId ] = properties[ propertyId ];

				return res;
			}, {} );
	}
);

export interface FilterPropertyControlProps {
	value: string;
	onChange: ( propertyId: string ) => void;
	properties: FilterProperties;
	propertyGroups: FilterPropertyGroups;
}

export const FilterPropertyControl = ( {
	onChange,
	value,
	properties,
	propertyGroups,
}: FilterPropertyControlProps ) => {
	return (
		// eslint-disable-next-line jsx-a11y/no-onchange
		<select
			className="gpeb-filter-property"
			onChange={ ( event ) => onChange( event.target.value ) }
			value={ value }
		>
			<option
				key={ 'select_property' }
				value={ '' }
				disabled={ true }
				hidden={ true }
			>
				– { __( 'Property', 'gp-entry-blocks' ) } –
			</option>

			{ /* Ungrouped Properties */ }
			{ Object.values(
				getPropertiesByGroup( properties, undefined )
			).map( ( groupProperty ) => (
				<option
					key={ groupProperty.value }
					value={ groupProperty.value }
				>
					{ groupProperty.label }
				</option>
			) ) }

			{ /* Grouped Properties */ }
			{ Object.entries( propertyGroups ).map(
				( [ propertyGroupId, propertyGroup ] ) => (
					<optgroup
						key={ propertyGroupId }
						label={ propertyGroup.label }
					>
						{ Object.values(
							getPropertiesByGroup( properties, propertyGroupId )
						).map( ( groupProperty ) => (
							<option
								key={ groupProperty.value }
								value={ groupProperty.value }
							>
								{ groupProperty.label }
							</option>
						) ) }
					</optgroup>
				)
			) }
		</select>
	);
};
