<?php

class PPW_Pro_QAL_Shortcode {
	/**
	 * Class instance
	 *
	 * @var PPWP_QAL_Shortcode
	 */
	private static $instance;

	/**
	 * Get service instance.
	 *
	 * @return PPWP_QAL_Shortcode|static
	 */
	public static function get_instance() {
		if ( is_null( self::$instance ) ) {
			// Use static instead of self due to the inheritance later.
			// For example: ChildSC extends this class, when we call get_instance
			// it will return the object of child class. On the other hand, self function
			// will return the object of base class.
			self::$instance = new static();
		}

		return self::$instance;
	}

	public function register() {
		add_shortcode( 'ppw_qal', array( $this, 'display_qal_shortcode' ) );
	}

	public function display_qal_shortcode( $attrs ) {
		$attrs = shortcode_atts(
			array(
				'key'                   => 'user_login',
				'link_text'             => 'click here',
				'based_on'              => 'label',
				'page_limit'            => 5,
				'error_message'         => 'Please login to view the content.',
				'format_time'           => 'F j, Y',
				'show_link'             => true,
				'show_title'            => true,
				'show_password'         => true,
				'show_usage_limit_left' => true,
				'show_post_thumbnail'   => true,
				'show_password_expiry' 	=> true,
				'empty_message'         => "You don't have any Quick Access Links.",
				'password_label'        => 'Password:',
				'password_expiry_label' => 'Password expiry:',
				'usage_limit_label'     => 'Usage limit (times left):',
				'access_link_label'     => 'Quick Access Link:',
				'empty_pwd_redirect'              => false,
			),
			$attrs
		);

		$current_user = wp_get_current_user();
		if ( ! $current_user->exists() ) {
			return sprintf( '<div class="ppwp-content__error">%s</div>', $attrs['error_message'] );
		}

		if ( ! isset( $attrs['key'] ) || ! isset( $current_user->{$attrs['key']} ) ) {
			return sprintf( '<div class="ppwp-content__error">%s</div>', "Invalid key param, please use user_login or ID." );
		}

		$repo      = new PPW_Pro_Repository();
		$passwords = [];
		switch ( $attrs['based_on'] ) {
			case 'user':
				if ( ! method_exists( 'PDA_Stats_PPW_Repository', 'fetch_passwords_by_username' ) ) {
					break;
				}
				$tracked_passwords = PDA_Stats_PPW_Repository::get_instance()->fetch_passwords_by_username( $current_user->user_login );
				if ( empty( $tracked_passwords ) ) {
					break;
				}
				$password_ids = array_reduce(
					$tracked_passwords,
					function ( $carry, $tracked_password ) {
						if ( ! in_array( $tracked_password->password_id, $carry ) ) {
							$carry[] = $tracked_password->password_id;
						}

						return $carry;
					},
					[]
				);
				if ( 0 === count( $password_ids ) ) {
					break;
				}

				$passwords = $repo->get_passwords_by_ids( $password_ids );
				break;
			case 'label':
				$passwords = $repo->get_password_info_by_label( $current_user->{$attrs['key']} );
				break;
			default:
				$passwords = [];
		}

		if ( empty( $passwords ) ) {
			if ( ( $attrs['empty_pwd_redirect'] !== false ) && ! is_admin() ) {
				if ( defined('REST_REQUEST') && REST_REQUEST === true ) {
					return null;
				}

				echo '<script>window.location="' . $attrs['empty_pwd_redirect'] . '"</script>';
				exit;
			}

			return "<div class='ppwp-content__error'>" . $attrs['empty_message'] . "</div>";
		}

		$links = array_map(
			function ( $info ) use ( $attrs ) {
				$link           = get_permalink( $info->post_id );
				$title          = get_the_title( $info->post_id );
				$post_thumbnail = '';

				if ( has_post_thumbnail( $info->post_id ) ) {
					$post_thumbnail = wp_get_attachment_image_src( get_post_thumbnail_id( $info->post_id ) );
				}

				if ( empty( $link ) ) {
					return '';
				}
				$param     = PPW_Pro_Constants::BYPASS_PARAM;
				$connector = '?';
				if ( strpos( $link, '?' ) > 0 ) {
					$connector = '&';
				}
				$param = $param . "=" . ppw_encrypt_decrypt( 'encrypt', [ 'password' => $info->password ] );

				$usage_limit_left = is_null( $info->usage_limit ) ? 'Unlimited' : $info->usage_limit - $info->hits_count;
				$expire_time      = is_null( $info->expired_date ) ? 'Never' : date( $attrs['format_time'], $info->expired_date );

				if ( is_array( $post_thumbnail ) && count( $post_thumbnail ) >= 3 ) {
					$post_thumbnail[0] = esc_attr( $post_thumbnail[0] );
					$post_thumbnail[1] = esc_attr( $post_thumbnail[1] );
					$post_thumbnail[2] = esc_attr( $post_thumbnail[2] );
				}

				return array(
					'id'             => $info->id,
					'link'           => wp_validate_boolean( $attrs['show_link'] ) ? esc_attr( "$link$connector$param" ) : false,
					'title'          => wp_validate_boolean( $attrs['show_title'] ) ? esc_attr( $title ) : false,
					'password'       => wp_validate_boolean( $attrs['show_password'] ) ? esc_attr( $info->password ) : false,
					'usageLimitLeft' => wp_validate_boolean( $attrs['show_usage_limit_left'] ) ? esc_attr( $usage_limit_left ) : false,
					'expireTime'     => wp_validate_boolean( $attrs['show_password_expiry'] ) ? esc_attr( $expire_time ) : false,
					'postThumbnail'  => wp_validate_boolean( $attrs['show_post_thumbnail'] ) ? $post_thumbnail : false,
				);
			},
			$passwords
		);

		$links = array_filter( $links, function( $link ) {
			return ! empty( $link );
		});
		$links = array_values( $links );

		if ( empty( $links ) ) {
			return "<div class='ppwp-content__error'>" . $attrs['empty_message'] . "</div>";
		}

		$labels = array_intersect_key(
			$attrs,
			array_flip(
				array( 'password_label', 'password_expiry_label', 'usage_limit_label', 'access_link_label' )
			)
		);

		wp_enqueue_script( "ppw-vue" );

		wp_enqueue_script(
			"ppw-qal-script",
			PPW_PRO_DIR_URL . "public/js/script.js",
			[],
			PPW_PRO_VERSION,
			'all'
		);

		ob_start();
		?>
		<div class="ppwp-qal-list" data-text="<?php echo esc_attr( $attrs['link_text'] ); ?>">
			<qal-list
					v-bind:limit="<?php echo absint( $attrs['page_limit'] ); ?>"
					v-bind:qals='<?php echo json_encode( $links ); ?>'
					v-bind:labels='<?php echo json_encode( $labels ); ?>'
			/>
		</div>
		<?php

		return ob_get_clean();
	}
}
