<?php

class PPW_Pro_Master_Password {
	/**
	 * @var PPW_Password_Services
	 */
	private $password_service;

	/**
	 * @var PPW_Repository_Passwords
	 */
	private $password_repository;

	/**
	 * @var PPW_Pro_Repository
	 */
	private $password_pro_repository;

	public function __construct() {
		$this->password_service        = new PPW_Password_Services();
		$this->password_repository     = new PPW_Repository_Passwords();
		$this->password_pro_repository = new PPW_Pro_Repository();
	}

	/**
	 * @var PPWP_Recaptcha
	 */
	protected static $instance;

	/**
	 * @return PPWP_Recaptcha
	 */
	public static function get_instance() {
		if ( null == self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	public function compatibility_with_free() {
		if ( ! defined( 'PPW_VERSION' ) ) {
			return false;
		}

		return version_compare( PPW_VERSION, '1.7.6', '>' );
	}

	/**
	 * Register hooks.
	 * @since 1.0.0
	 */
	public function register() {
		if ( ! $this->compatibility_with_free() ) {
			return;
		}

//		add_filter( 'ppwp_sitewide_has_bypass', array( $this, 'maybe_validate_sitewide_password' ), 100 );
		add_filter( 'ppw_shortcode_unlock_content', array( $this, 'maybe_unlock_shortcode_content' ), 100, 2 );
		add_filter( 'ppw_cookie_master_passwords', array( $this, 'maybe_handle_cookie_master_passwords' ), 100 );
		add_filter( 'ppw_pcp_api_result', array( $this, 'maybe_handle_pcp_api_result' ), 10, 3 );
		add_filter( 'ppw_allowed_master_protection_type', '__return_true' );
		add_filter( 'ppw_master_password_protection_types', array( $this, 'get_protection_types' ) );
		add_filter( 'ppw_pcp_area_is_valid_password', array( $this, 'maybe_valid_master_password' ), 20, 2 );
	}

//	public function maybe_validate_sitewide_password( $allowed ) {
//		if ( ! is_singular() ) {
//			return $allowed;
//		}
//
//		if ( ! isset( $_POST['input_wp_protect_password'] ) ) {
//			return $this->validate_master_cookie( $allowed, 'sitewide' );
//		}
//
//		$password = $_POST['input_wp_protect_password'];
//		$post_id  = get_the_ID();
//
//		return $this->validate_master_password_input( $password, 'sitewide', $allowed, $post_id );
//	}

	public function validate_protection_type( $protection_type, $protection_types ) {
		$protection_types = explode( ';', $protection_types );

		return in_array( $protection_type, $protection_types );
	}

	/**
	 * Check master cookie is exist.
	 *
	 */
	public function get_master_password_ids_from_cookie() {
		if ( ! isset( $_COOKIE ) ) {
			return [];
		}

		$master_password_ids = [];
		// Check with cookie name which contains master password name.
		foreach ( $_COOKIE as $key => $value ) {
			if ( false !== strpos( $key, PPW_Constants::MASTER_COOKIE_NAME ) ) {
				$master_password_id    = str_replace( PPW_Constants::MASTER_COOKIE_NAME, '', $key );
				$master_password_id    = str_replace( COOKIEHASH, '', $master_password_id );
				$master_password_ids[] = $master_password_id;
			}
		}

		return $master_password_ids;
	}

	public function validate_master_cookie( $allowed, $protection_type ) {
		$master_password_ids = $this->get_master_password_ids_from_cookie();
		if ( empty( $master_password_ids ) ) {
			return $allowed;
		}

		$post_id          = get_the_ID();
		$master_passwords = $this->password_pro_repository->get_activate_master_passwords_by_ids( $master_password_ids );
		if ( empty( $master_passwords ) ) {
			return $allowed;
		}

		$master_passwords = array_filter(
			$master_passwords,
			function ( $master_password ) use ( $protection_type ) {
				if ( ! isset( $master_password->protection_types ) ) {
					return false;
				}

				return $this->validate_protection_type( $protection_type, $master_password->protection_types );
			}
		);

		// Get all passwords which exist current post type.
		$master_passwords = $this->password_service->massage_master_passwords_with_post_type( $master_passwords, $post_id );

		if ( count( $master_passwords ) > 0 ) {
			// Valid master cookies.
			foreach ( $master_passwords as $master_password ) {
				if ( $this->password_service->is_valid_cookie( $master_password->id, array( $master_password->password ), PPW_Constants::MASTER_COOKIE_NAME ) ) {
					return true;
				}
			}
		}

		return $allowed;
	}

	/**
	 * @param $password
	 * @param $protection_type
	 * @param $allowed
	 * @param $post_id
	 *
	 * @return bool|mixed
	 */
	public function validate_master_password_input( $password, $protection_type, $allowed, $post_id ) {
		$master_password = $this->get_valid_master_password( $password, $post_id, $protection_type );
		if ( ! $master_password ) {
			return $allowed;
		}

		$current_roles = ppw_core_get_current_role();
		$result        = $this->password_service->check_valid_master_password( $master_password, $current_roles, $password );

		if ( $result['is_valid'] ) {
			/**
			 * Save cookie to client.
			 * If $result['role'] is not empty then password will be role password
			 * Else global password.
			 */
			$this->password_service->set_cookie_bypass_cache( $password . $result['role'] . $master_password->id, PPW_Constants::MASTER_COOKIE_NAME . $master_password->id );

			// Count when user enter right password.
			$this->password_repository->update_password(
				$master_password->id,
				array(
					'hits_count' => (int) $master_password->hits_count + 1,
				)
			);

			if ( ! empty( $_SERVER['HTTP_REFERER'] ) ) {
				$current_url = $_SERVER['HTTP_REFERER']; //phpcs:ignore
			} else {
				global $wp;
				$current_url = add_query_arg( $wp->query_string, '', home_url( $wp->request ) );
			}

			$current_url = apply_filters( 'ppwp_master_password_redirect_url', $current_url );

			wp_redirect( $current_url );

			return true;
		}

		return $allowed;
	}

	public function get_valid_master_password( $password, $post_id, $protection_type ) {
		$master_password = $this->password_repository->get_master_password_info_by_password( $password );
		if ( empty( $master_password ) ) {
			return false;
		}

		if ( ! isset( $master_password->protection_types ) ) {
			return false;
		}

		if ( ! $this->validate_protection_type( $protection_type, $master_password->protection_types ) ) {
			return false;
		}

		// Check post type is exist in password.
		if ( ! $this->password_service->check_post_type_for_master_password( $post_id, $master_password ) ) {
			return false;
		}

		return $master_password;
	}

	public function maybe_unlock_shortcode_content( $unlocked, $attrs ) {
		if ( ! empty( $attrs['section'] ) ) {
			// Handle Master Password for Content Protection Area.
			if ( ! $this->is_allowed_area() ) {
				return $unlocked;
			}
		} else {
			// Handle Master Password for PCP Shortcode default.
			if ( ! $this->is_allowed_pcp() ) {
				return $unlocked;
			}
		}

		return $this->validate_master_cookie( $unlocked, 'pcp' );
	}

	public function maybe_handle_cookie_master_passwords( $master_passwords ) {
		if ( empty( $master_passwords ) ) {
			return $master_passwords;
		}

		$filtered_master_passwords = array_filter(
			$master_passwords,
			function ( $master_password ) {
				if ( empty( $master_password->protection_types ) ) {
					return true;
				}

				return $this->validate_protection_type( 'single', $master_password->protection_types );
			}
		);

		return array_values( $filtered_master_passwords );
	}

	public function maybe_handle_pcp_api_result( $result, $password, $post ) {
		if ( ! $this->is_allowed_pcp() ) {
			return $result;
		}

		$master_password = $this->get_valid_master_password( $password, $post->ID, 'pcp' );
		if ( ! $master_password ) {
			return $result;
		}


		$current_roles   = ppw_core_get_current_role();
		$password_result = $this->password_service->check_valid_master_password( $master_password, $current_roles, $password );

		if ( $password_result['is_valid'] ) {
			// Count when user enter right password.
			$this->password_repository->update_password(
				$master_password->id,
				array(
					'hits_count' => (int) $master_password->hits_count + 1,
				)
			);

			$cookie_name = PPW_Constants::MASTER_COOKIE_NAME . $master_password->id . COOKIEHASH;
			$cookie_data = $this->password_service->generate_cookie_data( $cookie_name, $password . $password_result['role'] . $master_password->id );

			$result['isValidMaster']      = true;
			$result['masterCookieName']   = $cookie_name;
			$result['masterCookieValue']  = $cookie_data['value'];
			$result['masterCookieExpire'] = $cookie_data['expire'];

			$this->tracking_master_pcp_password( $password, $post->ID );
		}

		return $result;
	}

	public function tracking_master_pcp_password( $password, $post_id ) {
		$meta_data         = array(
			'protection_type' => 'pcp',
		);
		$meta_data         = apply_filters( 'ppwp_master_pcp_password_meta', $meta_data );
		$meta_data_encoded = wp_json_encode( $meta_data );
		$data              = array(
			'user_agent'  => ppw_pro_get_current_user_agent(),
			'ip_address'  => ppw_pro_get_current_ip_address(),
			'password'    => $password,
			'username'    => ppw_pro_get_current_user_name(),
			'post_type'   => 'ppwp_master',
			'access_date' => time(),
			'post_id'     => $post_id,
			'meta_data'   => $meta_data_encoded,
		);

		do_action( 'ppwp_track_password', $data );
	}

	public function filter_master_single_cookie( $master_passwords ) {
		return array_filter(
			$master_passwords,
			function ( $master_password ) {
				if ( ! isset( $master_password->protection_types ) ) {
					return false;
				}

				if ( empty( $master_password->protection_types ) ) {
					return true;
				}
				$protection_types = explode( ';', $master_password->protection_types );

				return in_array( 'single', $protection_types );
			}
		);
	}

	public function get_protection_types() {
		return [
			[
				'key'   => 'single',
				'value' => 'Individual',
			],
			[
				'key'   => 'pcp',
				'value' => 'Partial Content',
			],
		];
	}

	public function maybe_valid_master_password( $is_valid, $params ) {
		if ( ! $this->is_allowed_area() ) {
			return $is_valid;
		}

		$password = $params['password'];
		$post_id  = $params['post_id'];

		$master_password = $this->get_valid_master_password( $password, $post_id, 'pcp' );
		if ( ! $master_password ) {
			return $is_valid;
		}

		$current_roles   = ppw_core_get_current_role();
		$password_result = $this->password_service->check_valid_master_password( $master_password, $current_roles, $password );

		if ( $password_result['is_valid'] ) {
			// Count when user enter right password.
			$this->password_repository->update_password(
				$master_password->id,
				array(
					'hits_count' => (int) $master_password->hits_count + 1,
				)
			);

			$this->password_service->set_cookie_bypass_cache( $password . $password_result['role'] . $master_password->id, PPW_Constants::MASTER_COOKIE_NAME . $master_password->id );

			$this->tracking_master_pcp_password( $password, $post_id );

			return true;
		}

		return $is_valid;
	}

	public function is_allowed_pcp() {
		return apply_filters( 'ppwp_master_passwords_unlock_pcp', true );
	}

	public function is_allowed_area() {
		return apply_filters( 'ppwp_master_passwords_unlock_section', true );
	}
}
