<?php
if ( ! class_exists( 'PPW_Pro_Repository' ) ) {
	/**
	 * Connect database
	 *
	 * Class PPW_Pro_Repository
	 */
	class PPW_Pro_Repository {

		/**
		 * Password Protect WordPress Pro Table
		 *
		 * @var string
		 */
		private $table_name;

		/**
		 * @var object
		 */
		private $wpdb;

		const CACHE_GROUP = 'ppwp';

		public function __construct() {
			global $wpdb;
			$this->wpdb       = $wpdb;
			$this->table_name = $this->wpdb->prefix . PPW_Pro_Constants::TBL_NAME;
		}

		/**
		 * Insert new data
		 *
		 * @param array $data Information password.
		 *
		 * @return int|False
		 * @throws Exception
		 */
		public function insert( $data ) {
			if ( empty( $data['created_time'] ) ) {
				$now                  = new DateTime();
				$data['created_time'] = $now->getTimestamp();
			}

			return $this->wpdb->insert( $this->table_name, $data );
		}

		/**
		 * Update page/post status
		 *
		 * @param string|int $post_id The post ID.
		 * @param string     $status  true|false.
		 */
		public function update_page_post_status( $post_id, $status ) {
			// Clear cache before update status post.
			ppw_pro_clear_cache_by_id( $post_id );

			update_post_meta( $post_id, PPW_Pro_Constants::AUTO_GENERATE_PWD_META_DATA, $status );
		}

		/**
		 * Check page/post is protected
		 *
		 * @param $post_id
		 *
		 * @return bool
		 */
		public function is_protected_item( $post_id ) {
			return get_post_meta( $post_id, PPW_Pro_Constants::AUTO_GENERATE_PWD_META_DATA, true ) === "true";
		}

		/**
		 * Get all post id by password type
		 *
		 * @param $type
		 *
		 * @return mixed
		 */
		public function get_all_post_id_by_type( $type ) {
			return $this->wpdb->get_results( $this->wpdb->prepare( "SELECT * FROM $this->table_name WHERE campaign_app_type = %s", $type ) );
		}

		/**
		 * Get password by post id, password and not type
		 *
		 * @param $password
		 * @param $post_id
		 * @param $type
		 *
		 * @return mixed
		 */
		public function get_password_info( $password, $post_id, $type ) {
			return $this->wpdb->get_row( $this->wpdb->prepare( "SELECT * FROM $this->table_name WHERE BINARY password = %s AND post_id = %s AND campaign_app_type != %s", $password, $post_id, $type ) );
		}

		/**
		 * Update password for feature protect private pages
		 *
		 * @param $all_page_id
		 * @param $password
		 *
		 * @throws Exception
		 */
		function update_password_for_feature_protect_private_pages( $all_page_id, $password ) {
			$this->insert_or_update_password_type_is_common( $all_page_id, $password );
			$this->delete_page_post_un_selected( $all_page_id );
		}

		/**
		 * Delete page or post user un selected in feature protect private pages
		 *
		 * @param $all_page_id
		 */
		function delete_page_post_un_selected( $all_page_id ) {
			$all_post_selected = $this->get_all_post_id_by_type( PPW_Pro_Constants::CAMPAIGN_TYPE['COMMON'] );

			$selected_posts = array_map( function ( $post ) {
				return $post->post_id;
			}, $all_post_selected );

			$post_id_remove = array_diff( $selected_posts, $all_page_id );
			foreach ( $post_id_remove as $post_id ) {
				$this->wpdb->delete( $this->table_name, array(
					'post_id'           => $post_id,
					'campaign_app_type' => 'Common'
				) );
			}
		}

		/**
		 * Check condition before insert or update password type is common
		 *
		 * @param $all_page_id
		 * @param $password
		 *
		 * @throws Exception
		 */
		function insert_or_update_password_type_is_common( $all_page_id, $password ) {
			foreach ( $all_page_id as $page_id ) {
				$advance_password = $this->get_password_by_post_id_and_type( $page_id, PPW_Pro_Constants::CAMPAIGN_TYPE['COMMON'] );
				// Check before insert or update password
				if ( is_null( $advance_password ) ) {
					$result = $this->insert(
						array(
							'post_id'           => $page_id,
							'password'          => $password,
							'campaign_app_type' => PPW_Pro_Constants::CAMPAIGN_TYPE['COMMON'],
						)
					);
				} else {
					$result = $this->wpdb->update(
						$this->table_name,
						array( 'password' => $password ),
						array( 'id' => $advance_password->id )
					);
				}

				if ( false === $result ) {
					send_json_data_error( __( PPW_Constants::BAD_REQUEST_MESSAGE, 'password-protect-page' ) );
				}

				// Check and protect page/post
				$password_services = new PPW_Pro_Password_Services();
				if ( ! $password_services->is_protected_content( $page_id ) ) {
					$this->update_page_post_status( $page_id, 'true' );
				}
			}
		}

		/**
		 * Get password by post id and type
		 *
		 * @param $post_id
		 * @param $type
		 *
		 * @return mixed
		 */
		public function get_password_by_post_id_and_type( $post_id, $type ) {
			return $this->wpdb->get_row( $this->wpdb->prepare( "SELECT * FROM $this->table_name WHERE post_id = %s AND campaign_app_type = %s", $post_id, $type ) );
		}

		/**
		 * Delete all password type is common
		 */
		public function delete_all_password_type_is_common() {
			$all_post_selected = $this->get_all_post_id_by_type( PPW_Pro_Constants::CAMPAIGN_TYPE['COMMON'] );
			foreach ( $all_post_selected as $post ) {
				$this->wpdb->delete(
					$this->table_name,
					array(
						'id' => $post->id
					)
				);
			}
		}

		/**
		 * Get password and password type by post id and campaign type
		 *
		 * @param $post_id
		 *
		 * @return mixed
		 */
		public function get_type_and_password_by_post_id_and_campaign_type( $post_id ) {
			$advance_password = $this->wpdb->get_results( $this->wpdb->prepare( "SELECT campaign_app_type, password FROM $this->table_name WHERE post_id = %s AND is_activated = 1 AND (expired_date IS NULL OR expired_date > UNIX_TIMESTAMP()) AND (usage_limit IS NULL OR hits_count < usage_limit) AND (campaign_app_type LIKE %s OR campaign_app_type = %s OR campaign_app_type = %s OR campaign_app_type = %s) ", $post_id, PPW_Pro_Constants::CAMPAIGN_TYPE['ROLE'] . '%', PPW_Pro_Constants::CAMPAIGN_TYPE['AUTO'], PPW_Pro_Constants::CAMPAIGN_TYPE['DEFAULT'], PPW_Pro_Constants::CAMPAIGN_TYPE['COMMON'] ) );

			return $advance_password;
		}

		/**
		 * Get all passwords by post_id
		 *
		 * @param $post_id
		 *
		 * @return mixed
		 */
		public function get_all_password_by_post_id( $post_id ) {
			$query_string = $this->wpdb->prepare( "SELECT * FROM $this->table_name WHERE post_id = %s AND (expired_date IS NULL OR expired_date > UNIX_TIMESTAMP()) AND (usage_limit IS NULL OR hits_count < usage_limit) AND is_activated = 1", $post_id );

			return $this->wpdb->get_row( $query_string );
		}

		/**
		 * Get advance password by password and post id
		 *
		 * @param $password
		 * @param $post_id
		 *
		 * @return mixed
		 */
		public function get_advance_password_by_password_and_post_id( $password, $post_id ) {
			$advance_password = $this->wpdb->get_row( $this->wpdb->prepare( "SELECT * FROM $this->table_name WHERE BINARY password = %s AND post_id = %s", $password, $post_id ) );

			return $advance_password;
		}

		/**
		 * get password by contact id
		 *
		 * @param $contact_id
		 *
		 * @return mixed
		 */
		public function get_password_by_contact_id( $contact_id ) {
			$query_string = $this->wpdb->prepare( "SELECT password FROM $this->table_name WHERE contact_id = %d and is_activated = 1", $contact_id );

			return $this->wpdb->get_row( $query_string );
		}

		/**
		 * update password by contact id
		 *
		 * @param $contact_id
		 * @param $data
		 *
		 * @return mixed
		 */
		public function update_password_by_contact_id( $contact_id, $data ) {
			return $this->wpdb->update( $this->table_name, $data, array(
				'contact_id' => $contact_id
			) );
		}

		/**
		 *
		 * Get password by post_id
		 * @since 1.3.0.1 Add $cached param
		 *
		 * @param string $post_id Post ID.
		 * @param bool   $cached   Using cache on the loop (One request).
		 *
		 * @return array Array passwords.
		 */
		public function get_password_by_post_id( $post_id, $cached = false ) {
			$passwords = [];
			$cache_key = 'ppw_passwords_post_id_' . $post_id;

			if ( $cached ) {
				$passwords = wp_cache_get( $cache_key );
			}

			if ( empty( $passwords ) ) {
				$query_string = $this->wpdb->prepare( "SELECT password FROM $this->table_name WHERE post_id = %s and is_activated = 1", $post_id );
				$passwords    = $this->wpdb->get_results( $query_string );
				if ( $cached ) {
					wp_cache_set( $cache_key, $passwords );
				}
			}

			return array_map(
				function ( $pass ) {
					return $pass->password;
				},
				$passwords
			);
		}

		/**
		 * Get password info by password and post id
		 *
		 * @param string     $password the password.
		 * @param string|int $post_id  the post id.
		 * @return mixed
		 */
		public function get_password_info_by_password_and_post_id( $password, $post_id ) {
			$query_string = $this->wpdb->prepare( "SELECT * FROM $this->table_name WHERE BINARY password = %s and is_activated = 1 and (expired_date is NULL OR expired_date > UNIX_TIMESTAMP()) and (usage_limit is NULL OR hits_count < usage_limit) and post_id = %s", $password, $post_id );

			return $this->wpdb->get_row( $query_string );
		}

		/**
		 * Get password info by password and post ID.
		 *
		 * @param string     $type The password type.
		 * @param string|int $post_id  The post id.
		 * @return mixed
		 */
		public function get_password_by_type( $type, $post_id ) {
			$query_string = $this->wpdb->prepare( "SELECT * FROM $this->table_name WHERE campaign_app_type = %s and is_activated = 1 and (expired_date is NULL OR expired_date > UNIX_TIMESTAMP()) and (usage_limit is NULL OR hits_count < usage_limit) and post_id = %s", $type, $post_id );

			return $this->wpdb->get_results( $query_string );
		}

		/**
		 * Get password info by password and post id
		 *
		 * @param string     $label
		 * @return mixed
		 */
		public function get_password_info_by_label( $label ) {
			$query_string = $this->wpdb->prepare( "SELECT * FROM $this->table_name WHERE is_activated = 1 and (expired_date is NULL OR expired_date > UNIX_TIMESTAMP()) and (usage_limit is NULL OR hits_count < usage_limit) and label = %s ORDER BY created_time DESC", $label );

			return $this->wpdb->get_results( $query_string );
		}

		/**
		 * get advance password by password
		 *
		 * @param $password
		 * @param $post_id
		 *
		 * @return mixed
		 */
		public function get_advance_password_by_password( $password, $post_id ) {
			$query_string = $this->wpdb->prepare( "SELECT * FROM $this->table_name WHERE BINARY password = %s and post_id = %s and is_activated = 1", $password, $post_id );

			return $this->wpdb->get_row( $query_string );
		}

		public function update_data_password_by_id( $id, $data ) {
			try {
				return $this->wpdb->update( $this->table_name, $data, array(
					'ID' => $id
				) );
			} catch ( Exception $e ) {

				return false;
			}
		}

		/**
		 * Get all id child page
		 *
		 * @param $page_id
		 *
		 * @return array
		 */
		public function get_all_id_child_page( $page_id ) {
			$page_list = [];
			$too_look  = [ $page_id ];
			while ( count( $too_look ) > 0 ) {
				$parent      = array_pop( $too_look );
				$page_list[] = $parent;
				$children    = $this->get_children( $parent );
				if ( empty( $children ) ) {
					continue;
				}

				// Add children of a page to get grandchildren.
				foreach ( $children as $child ) {
					$too_look[] = $child;
				}
			}
			// Remove page_id added.
			array_shift( $page_list );

			return $page_list;
		}

		/**
		 * Get children of a page.
		 *
		 * @param integer $page_id Page ID.
		 *
		 * @return array
		 */
		public function get_children( $page_id ) {
			$key   = 'post_children_' . $page_id;
			$value = wp_cache_get( $key, self::CACHE_GROUP );

			if ( false !== $value ) {
				return $value;
			}

			$args = array(
				'posts_per_page' => - 1,
				'order'          => 'DESC',
				'post_parent'    => $page_id,
				'post_type'      => get_post_type( $page_id ),
			);

			$children = get_children( $args, ARRAY_A );
			$children = array_map(
				function ( $page_child ) {
					return $page_child['ID'];
				},
				$children
			);
			wp_cache_set( $key, $children, self::CACHE_GROUP );

			return $children;
		}

		/**
		 * Get password info by post id
		 *
		 * @param $post_id
		 *
		 * @return mixed
		 */
		public function get_password_info_by_post_id( $post_id ) {
			$query_string = $this->wpdb->prepare( "SELECT * FROM $this->table_name WHERE post_id = %s", $post_id );

			return $this->wpdb->get_results( $query_string );
		}

		/**
		 * @param integer $post_id
		 * @param array   $options
		 *
		 * @return mixed
		 */
		public function get_passwords( $post_id, $options ) {
			$args   = wp_parse_args(
				$options,
				[
					'order_by'  => 'id',
					'order'     => 'desc',
					'page'      => 0,
					'page_size' => 0,
				]
			);
			$offset = ( $args['page'] - 1 ) * $args['page_size'];
			$limit  = $args['page_size'];

			$query_string = $this->wpdb->prepare( "
				SELECT * FROM $this->table_name
					WHERE post_id = %d 
					ORDER BY {$args['order_by']} {$args['order']}
					LIMIT %d OFFSET %d"
				, absint( $post_id ), $limit, $offset
			);

			return $this->wpdb->get_results( $query_string );
		}

		/**
		 * @param integer $post_id
		 * @param array   $options
		 *
		 * @return mixed
		 */
		public function find_passwords( $post_id, $term, $options ) {
			$args   = wp_parse_args(
				$options,
				[
					'order_by'  => 'id',
					'order'     => 'desc',
					'page'      => 0,
					'page_size' => 0,
					'searchBy'  => 'password',
				]
			);
			$offset = ( $args['page'] - 1 ) * $args['page_size'];
			$limit  = $args['page_size'];

			$query_string = $this->wpdb->prepare( "
				SELECT * FROM $this->table_name
					WHERE post_id = %d AND {$args['searchBy']} LIKE %s
					ORDER BY {$args['order_by']} {$args['order']}
					LIMIT %d OFFSET %d"
				, absint( $post_id ),
				'%' . $term . '%',
				$limit,
				$offset
			);

			return $this->wpdb->get_results( $query_string );
		}

		/**
		 * @param $post_id
		 *
		 * @return mixed
		 */
		public function count_passwords( $post_id ) {
			$query_string = $this->wpdb->prepare( "SELECT COUNT(*) FROM $this->table_name WHERE post_id = %s", $post_id );

			return $this->wpdb->get_var( $query_string );
		}

		/**
		 * @param        $post_id
		 * @param        $term
		 * @param string $search_by
		 *
		 * @return mixed
		 */
		public function count_searched_passwords( $post_id, $term, $search_by = 'password' ) {
			$query_string = $this->wpdb->prepare( "SELECT COUNT(*) FROM $this->table_name WHERE post_id = %d AND {$search_by} LIKE %s", absint( $post_id ), '%' . $term . '%' );

			return $this->wpdb->get_var( $query_string );
		}

		/**
		 * Delete password by id
		 *
		 * @param $id
		 *
		 * @return mixed
		 */
		function delete_password_by_id( $id ) {
			return $this->wpdb->delete(
				$this->table_name,
				array(
					'ID' => $id
				)
			);
		}

		/**
		 * Get password info by post id
		 *
		 * @param $post_id
		 *
		 * @return mixed
		 */
		function delete_password_by_post_id( $post_id ) {
			return $this->wpdb->delete(
				$this->table_name,
				array(
					'post_id' => $post_id
				)
			);
		}

		/**
		 * Delete selected passwords by id
		 * String will convert to int
		 *
		 * @param array $selected_ids ID Passwords selected.
		 *
		 * @return mixed
		 */
		public function delete_selected_passwords( $selected_ids ) {
			$selected_ids = implode( ',', array_map( 'absint', $selected_ids ) );

			return $this->wpdb->query( "DELETE FROM $this->table_name WHERE ID IN($selected_ids)" );
		}

		/**
		 * Display all allowed password type.
		 */
		private function get_allowed_password_type() {
			$default_type        = PPW_Pro_Constants::CAMPAIGN_TYPE['DEFAULT'];
			$common_type         = PPW_Pro_Constants::CAMPAIGN_TYPE['COMMON'];
			$auto_type           = PPW_Pro_Constants::CAMPAIGN_TYPE['AUTO'];
			$emai_marketing_type = PPW_Pro_Constants::CAMPAIGN_TYPE['ACTIVE_CAMPAIGN'];

			$types = array(
				"'$default_type'",
				"'$common_type'",
				"'$auto_type'",
				"'$emai_marketing_type'"
			);

			$types = apply_filters( 'ppwp_allowed_password_type', $types );

			return implode( ', ', $types );
		}

		/**
		 * Get all password by campaign app type
		 *
		 * @return mixed
		 */
		public function get_all_password_by_campaign_app_type() {
			$allowed_types = $this->get_allowed_password_type();
			$role_type     = PPW_Pro_Constants::CAMPAIGN_TYPE['ROLE'];

			return $this->wpdb->get_results( "SELECT * FROM $this->table_name WHERE campaign_app_type LIKE '$role_type%' OR campaign_app_type IN ($allowed_types)" );
		}

		/**
		 * @return mixed
		 */
		public function get_all_protected_posts() {
			$table_name = $this->wpdb->prefix . 'postmeta';
			// Check Exist
			$result = $this->wpdb->get_results( "SELECT DISTINCT post_id FROM $table_name WHERE meta_key = '" . PPW_Pro_Constants::AUTO_GENERATE_PWD_META_DATA . "' AND meta_value = 'true' " );

			return $result;

		}

		/**
		 * Delete password by post id and is default
		 *
		 * @param $post_id
		 * @param $is_default
		 *
		 * @return mixed
		 */
		public function delete_password_by_post_id_and_is_default( $post_id, $is_default ) {
			return $this->wpdb->delete( $this->table_name,
				array(
					'post_id'    => $post_id,
					'is_default' => $is_default
				)
			);
		}

		/**
		 * Delete password by post id and password
		 *
		 * @param $post_id
		 * @param $password
		 *
		 * @return mixed
		 */
		function delete_password_by_post_id_and_password( $post_id, $password ) {
			return $this->wpdb->delete( $this->table_name,
				array(
					'post_id'  => $post_id,
					'password' => $password
				)
			);
		}

		/**
		 * Get passwords by post_id
		 *
		 * @param $post_id
		 *
		 * @return array
		 */
		public function get_passwords_by_post_id( $post_id ) {
			$query_string = $this->wpdb->prepare( "SELECT password FROM $this->table_name WHERE post_id = %s", $post_id );

			return $this->wpdb->get_col( $query_string );
		}

		/**
		 * Get active passwords by id and types.
		 * String will convert to int
		 *
		 * @param array  $ids  ID Passwords selected.
		 * @param string $type Type password.
		 *
		 * @return array|object|null Database query results
		 */
		public function fetch_pcp_passwords_by_ids_and_type( $ids, $type ) {
			$ids_str = implode( ',', array_map( 'absint', $ids ) );

			$query = $this->wpdb->prepare( "
				SELECT * FROM $this->table_name 
				WHERE ID IN($ids_str)
				AND campaign_app_type = %s
				AND (expired_date IS NULL OR expired_date > UNIX_TIMESTAMP()) 
				AND (usage_limit IS NULL OR hits_count < usage_limit) 
				AND is_activated = 1",
				$type );

			return $this->wpdb->get_results( $query, ARRAY_A );
		}

		/**
		 * Get activate passwords by id and types.
		 * String will convert to int
		 *
		 * @param string $type Type password.
		 *
		 * @return array|object|null Database query results
		 */
		public function fetch_passwords_by_type( $type ) {
			$query = $this->wpdb->prepare( "SELECT * FROM $this->table_name where campaign_app_type LIKE %s ORDER BY created_time DESC", $type . '%');

			return $this->wpdb->get_results( $query );
		}

		/**
		 * Fetch label by password content type.
		 *
		 * @param string $type Password content type (referer to defined values in PPW_Pro_Constants::CAMPAIGN_TYPE)
		 *
		 * @return array The query results.
		 */
		public function fetch_labels_by_content_type( $type ) {
			$query = $this->wpdb->prepare( "SELECT DISTINCT (BINARY label) as label FROM $this->table_name  where campaign_app_type LIKE %s AND label IS NOT NULL", $type . '%' );

			return $this->wpdb->get_results( $query );
		}

		/**
		 * Fetch all active PCP passwords by global and roles
		 *
		 * @param array $ids              Array passwords ids.
		 * @param bool  $is_check_expired Is check expired (date, count).
		 *
		 * @return array|object|null Database query results
		 */
		public function fetch_activate_pcp_passwords_by_ids( $ids, $roles, $is_check_expired = false ) {
			$like_where    = $this->generate_where_like_for_roles( $roles );
			$expired_where = '';
			if ( $is_check_expired ) {
				$expired_where = " AND (expired_date IS NULL OR expired_date > UNIX_TIMESTAMP()) AND (usage_limit IS NULL OR hits_count < usage_limit) ";
			}
			$ids_str = implode( ',', array_map( 'absint', $ids ) );
			$query   = $this->wpdb->prepare( "
				SELECT * FROM $this->table_name 
				WHERE ID IN($ids_str)
				{$expired_where}
				AND is_activated = 1
				AND ( campaign_app_type = %s {$like_where})",
				PPW_Pro_Constants::CAMPAIGN_TYPE['SHORTCODE']
			);

			return $this->wpdb->get_results( $query );
		}

		/**
		 * Fetch all activate PCP passwords by labels.
		 *
		 * @param array $labels              Array labels.
		 * @param array $roles               Current user roles.
		 * @param bool  $should_check_expiry When it's true, the function will append the where condition to check the expiry (usage_limit, expired_date).
		 * @param array $user_passwords      User Passwords.
		 *
		 * @return array|object|null Database query results
		 */
		public function fetch_activate_pcp_passwords_by_labels( $labels, $roles, $should_check_expiry, $user_passwords ) {
			if ( empty( $labels ) ) {
				return array();
			}
			$user_passwords = array_map( function ( $user_password ) {
				$user_password = wp_unslash( $user_password );
				$user_password = esc_sql( $user_password );

				return "'{$user_password}'";
			}, $user_passwords );

			$client_password = implode( ',', $user_passwords );
			$extra           = "AND binary password IN ({$client_password})";

			$like_where    = $this->generate_where_like_for_roles( $roles );
			$expired_where = '';
			if ( $should_check_expiry ) {
				$expired_where = " AND (expired_date IS NULL OR expired_date > UNIX_TIMESTAMP()) AND (usage_limit IS NULL OR hits_count < usage_limit) ";
			}

			$lbl_str = implode( ',',
				array_map(
					function ( $label ) {
						return "'$label'";
					},
					$labels
				)
			);

			$query = $this->wpdb->prepare( "
				SELECT * FROM $this->table_name 
				WHERE binary label IN({$lbl_str})  
				{$expired_where}
				AND is_activated = 1
				AND ( campaign_app_type = %s {$like_where})
				{$extra}",
				PPW_Pro_Constants::CAMPAIGN_TYPE['SHORTCODE']
			);

			return $this->wpdb->get_results( $query );
		}

		public function find_activated_pcp_password( $password, $labels, $roles) {
			if ( empty( $labels ) ) {
				return array();
			}

			$like_where    = $this->generate_where_like_for_roles( $roles );
			$expired_where = " AND (expired_date IS NULL OR expired_date > UNIX_TIMESTAMP()) AND (usage_limit IS NULL OR hits_count < usage_limit) ";

			$lbl_str = implode( ',',
				array_map(
					function ( $label ) {
						return "'$label'";
					},
					$labels
				)
			);

			$query = $this->wpdb->prepare( "
				SELECT * FROM $this->table_name 
				WHERE BINARY label IN({$lbl_str})  
				{$expired_where}
				AND BINARY password = %s
				AND is_activated = 1
				AND ( campaign_app_type = %s {$like_where})
				",
				$password,
				PPW_Pro_Constants::CAMPAIGN_TYPE['SHORTCODE']
			);

			return $this->wpdb->get_row( $query );
		}

		public function find_activated_pcp_password_by_ids( $password_ids, $labels, $roles ) {
			$password_ids     = array_map( 'absint', $password_ids );
			$password_ids_str = implode( ', ', $password_ids );
			if ( empty( $labels ) ) {
				return array();
			}

			$like_where    = $this->generate_where_like_for_roles( $roles );
			$expired_where = " AND (expired_date IS NULL OR expired_date > UNIX_TIMESTAMP()) AND (usage_limit IS NULL OR hits_count < usage_limit) ";

			$lbl_str = implode( ',',
				array_map(
					function ( $label ) {
						return "'$label'";
					},
					$labels
				)
			);

			$query = $this->wpdb->prepare( "
				SELECT * FROM $this->table_name 
				WHERE BINARY label IN({$lbl_str})  
				{$expired_where}
				AND id IN ({$password_ids_str})
				AND is_activated = 1
				AND ( campaign_app_type = %s {$like_where})
				",
				PPW_Pro_Constants::CAMPAIGN_TYPE['SHORTCODE']
			);

			return $this->wpdb->get_results( $query );
		}

		/**
		 * Generate query to get password roles type in DB
		 *
		 * @param array  $roles    User roles.
		 *
		 * @return string
		 */
		private function generate_where_like_for_roles( $roles ) {
			$where_like_string = '';
			$pcp_role          = PPW_Pro_Constants::CAMPAIGN_TYPE['SHORTCODE_ROLE'];
			if ( is_array( $roles ) && count( $roles ) > 0 ) {
				/**
				 * Generate roles to string with like condition.
				 * Example:
				 *    ['editor,'admin'] to ' OR campaign_app_type LIKE '%editor% OR campaign_app_type LIKE '%admin%'
				 */
				$where_like_string = array_reduce(
					$roles,
					function ( $carry, $role ) use ( $pcp_role ) {
						if ( ! empty( $role ) ) {
							$carry = $carry . "OR campaign_app_type LIKE '%{$pcp_role}{$role};%' OR campaign_app_type LIKE '%{$pcp_role}{$role}' ";
						}

						return $carry;
					}, $where_like_string );
				$where_like_string = ! empty( $where_like_string ) ? $where_like_string : '';
			}

			return $where_like_string;
		}

		/**
		 * Get PCP Passwords data.
		 *
		 * @param string $password Password.
		 *
		 * @return array|object|void|null Database query result in format specified by $output or null on failure
		 */
		public function get_pcp_password( $password ) {
			$sql = $this->wpdb->prepare( "SELECT * FROM {$this->table_name} WHERE BINARY password = %s AND campaign_app_type LIKE %s", $password, PPW_Pro_Constants::CAMPAIGN_TYPE['SHORTCODE'] . '%'  );

			return $this->wpdb->get_row( $sql );
		}

		/**
		 * Add a row in table by id.
		 *
		 * @param array $data Data to add.
		 *
		 * @return int|false The number of rows updated, or false on error.
		 */
		public function add_new_password( $data ) {
			$is_added = $this->wpdb->insert( $this->table_name, $data );
			if ( $is_added ) {
				return $this->wpdb->insert_id;
			}

			return false;
		}

		/**
		 * Update count by password
		 *
		 * @param string $password
		 *
		 * @return bool
		 */
		public function update_count_by_password( $password ) {
			$sql = $this->wpdb->prepare( "UPDATE {$this->table_name} SET hits_count = hits_count + 1 WHERE BINARY password = %s AND campaign_app_type LIKE %s", $password, PPW_Pro_Constants::CAMPAIGN_TYPE['SHORTCODE'] . '%' );

			return $this->wpdb->get_row( $sql );
		}

		/**
		 * Update count by password
		 *
		 * @param string $password
		 *
		 * @return bool
		 */
		public function update_count_by_password_id( $password_id ) {
			$sql = $this->wpdb->prepare( "UPDATE {$this->table_name} SET hits_count = hits_count + 1 WHERE id = %d AND campaign_app_type LIKE %s", $password_id, PPW_Pro_Constants::CAMPAIGN_TYPE['SHORTCODE'] . '%' );

			return $this->wpdb->get_row( $sql );
		}

		/**
		 * Find password by hash.
		 *
		 * @param string $hash    Password hashed.
		 * @param array  $roles   Current user roles.
		 * @param int    $post_id The post ID user enters password.
		 * @param bool   $cached  Whether to use wp_cache in this query.
		 *
		 * @return array
		 */
		public function find_password_by_hash( $hash, $roles, $post_id, $cached = true ) {
			$cache_key = 'ppw_passwords_hash_' . $hash;
			$passwords = [];

			if ( $cached ) {
				$passwords = wp_cache_get( $cache_key );
			}

			if ( ! empty( $passwords ) ) {
				return $passwords;
			}

			$sql = $this->wpdb->prepare( "SELECT * FROM {$this->table_name} WHERE post_id = %d AND is_activated = 1", $post_id );
			$sql = $this->append_query_condition( $sql, $hash, $roles );

			if ( empty( $sql ) ) {
				return $passwords;
			}

			$passwords = $this->wpdb->get_row( $sql );
			if ( $cached ) {
				wp_cache_set( $cache_key, $passwords );
			}

			return $passwords;
		}

		/**
		 * Create global pwd query string by checking
		 *  + campaign_app_type is Global
		 *  + hash password and post_id
		 *
		 * @param string $hash  MD5 hashed string of password and post ID.
		 * @param array  $types The password types.
		 *
		 * @return string
		 *      Empty if there is no supported password types.
		 *      Non-empty string example:
		 *          ( campaign_app_type IN ( 'Default','Auto','ActiveCampaign','Common','WooCommerce' ) AND MD5(CONCAT(password, post_id)) = 'hashed-value' )
		 */
		private function create_pwd_query_with_types( $hash, $types ) {
			if ( empty( $types ) ) {
				return '';
			}

			$str_types = implode( ',',
				array_map( function ( $type ) {
					return "'${type}'";
				}, $types
				)
			);

			$type_comparision = "campaign_app_type IN ( ${str_types} )";

			return $this->wpdb->prepare( "( ${type_comparision} AND MD5(CONCAT(password, post_id)) = %s )", $hash );
		}

		/**
		 * Create role pwd query string by checking
		 *   + hash password, role and post ID.
		 *
		 * @param string $hash  MD5 hashed string of password and post ID.
		 * @param array  $roles Current user roles.
		 *
		 * @return string
		 *     Empty means there is no rules
		 *     Non-empty example string: MD5(CONCAT(password, 'editor', post_id)) = 'hashed-value' || MD5(CONCAT(password, 'subscriber', post_id)) = 'hashed-value'
		 */
		private function create_role_pwd_query( $hash, $roles ) {
			$roles_query = array_map( function ( $role ) use ( $hash ) {
				return $this->wpdb->prepare( 'MD5(CONCAT(password, %s, post_id)) = %s', $role, $hash );
			}, $roles );

			if ( empty( $roles_query ) ) {
				$roles_query_string = '';
			} else {
				$roles_query_string = implode( ' || ', $roles_query );
			}

			return $roles_query_string;
		}

		/**
		 * Append query condition based on password types and user roles.
		 *
		 * @param string $sql The current SQL string.
		 * @param string $hash Hashed strint to compare.
		 * @param array $roles The current user roles.
		 *
		 * @return string
		 *   Empty string means there is no supported types and roles.
		 */
		private function append_query_condition( $sql, $hash, $roles ) {
			// Get password types excluding role type.
			$pwd_types               = array_keys( PPW_Pro_Constants::get_map_pwd_types() );
			$global_pwd_query_string = $this->create_pwd_query_with_types( $hash, $pwd_types );
			$roles_query_string = $this->create_role_pwd_query( $hash, $roles );

			$query_condition = '';
			if ( ! empty( $global_pwd_query_string ) && ! empty( $roles_query_string ) ) {
				$query_condition .= " AND ( $global_pwd_query_string || $roles_query_string )";
				return $sql . $query_condition;
			}

			if ( ! empty( $global_pwd_query_string ) ) {
				$query_condition .= " AND $global_pwd_query_string";
				return $sql . $query_condition;
			}

			if ( ! empty( $roles_query_string ) ) {
				$query_condition .= " AND ( $roles_query_string )";
				return $sql . $query_condition;
			}

			return $query_condition;
		}


		/**
		 * @param $ids array Array ID.
		 *
		 * @return array|object|null Database query results.
		 */
		public function get_activate_master_passwords_by_ids( $ids ) {
			$ids          = array_map( 'absint', $ids );
			$ids_value    = implode( ',', $ids );
			$query_string = "SELECT * FROM {$this->table_name} WHERE post_id = 0 AND campaign_app_type LIKE 'master_%' and is_activated = 1 and id IN ({$ids_value})";

			return $this->wpdb->get_results( $query_string );
		}

		/**
		 * @param $ids array Array ID.
		 *
		 * @return array|object|null Database query results.
		 */
		public function get_passwords_by_ids( $ids ) {
			$ids          = array_map( 'absint', $ids );
			$ids_value    = implode( ',', $ids );
			$query_string = "SELECT * FROM {$this->table_name} WHERE id IN ({$ids_value}) ORDER BY FIELD(id,{$ids_value});";

			return $this->wpdb->get_results( $query_string );
		}
		
		public function ppw_increment_hits_count($post_id, $pass, $hits_count) {
			$sql = $this->wpdb->prepare( "
				UPDATE {$this->table_name} 
				SET hits_count = %d + 1 
				WHERE BINARY password = %s
				AND post_id = %d", 
				$hits_count, $pass, $post_id );
		
			return $this->wpdb->get_row( $sql );
		}
	}
}
