<?php
if ( ! class_exists( "PPW_Pro_Api" ) ) {
	/**
	 * Class WP_Protect_Password_Api
	 */
	class PPW_Pro_Api {
		/**
		 * @var PPW_Pro_Password_Services
		 */
		private $service;

		/**
		 * PPW_Pro_Api constructor.
		 */
		public function __construct() {
			$this->service = new PPW_Pro_Password_Services();
		}

		/**
		 * Register rest routes
		 */
		public function register_rest_routes() {
			#region API Meta Box
			register_rest_route( 'wppp/v1', '/get-data-password/(?P<id>\d+)', array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_password' ),
				'permission_callback' => array( $this, 'can_access' ),
			) );

			register_rest_route( 'wppp/v1', '/add-new-password/(?P<id>\d+)', array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'add_new_password' ),
				'permission_callback' => array( $this, 'can_access' ),
			) );

			register_rest_route( 'wppp/v1', '/create-multiple-passwords/(?P<id>\d+)', array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'create_multiple_passwords' ),
				'permission_callback' => array( $this, 'can_access' ),
			) );

			register_rest_route( 'wppp/v1', '/file-is-protect/(?P<id>\d+)', array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'file_is_protect' ),
				'permission_callback' => array( $this, 'can_access' ),
			) );

			register_rest_route( 'wppp/v1', '/protect-this-file/(?P<id>\d+)', array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'protect_this_post' ),
				'permission_callback' => array( $this, 'can_access' ),
			) );

			register_rest_route( 'wppp/v1', '/un-protect-this-file/(?P<id>\d+)', array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'un_protect_this_post' ),
				'permission_callback' => array( $this, 'can_access' ),
			) );
			#endregion

			register_rest_route( 'wppp/v1', '/get-all-password/(?P<id>\d+)', array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_all_password' ),
				'permission_callback' => array( $this, 'can_access' ),
			) );

			register_rest_route( 'wppp/v1', '/passwords/(?P<id>\d+)', array(
					'methods'             => 'GET',
					'callback'            => array( $this, 'get_passwords' ),
					'permission_callback' => array( $this, 'can_access' ),
					'args'                => [
						'term'     => array(
							'default' => '',
							'type'    => 'string',
						),
						'searchBy' => array(
							'enum' => array(
								'password',
								'label',
								'qal_slug',
							),
							'type' => 'string',
						),
						'id'       => array(
							'description'       => __( 'Post ID.' ),
							'sanitize_callback' => 'absint',
							'type'              => 'number',
						),
						'pageSize' => array(
							'description'       => __( 'Page size the result set by a specific number of items.' ),
							'sanitize_callback' => 'absint',
							'type'              => 'number',
						),
						'page'     => array(
							'description'       => __( 'Page the result set by a specific current page.' ),
							'sanitize_callback' => 'absint',
							'type'              => 'number',
						),
						'orderBy'  => array(
							'default'     => 'id',
							'description' => __( 'Sort collection by object attribute.' ),
							'enum'        => array(
								'id',
								'password',
								'label',
								'campaign_app_type',
								'is_activated',
								'created_time',
								'expired_date',
								'hits_count',
								'usage_limit',
								'qal_slug',
							),
							'type'        => 'string',
						),
						'order'    => array(
							'default'     => 'desc',
							'description' => __( 'Order sort attribute ascending or descending.' ),
							'enum'        => array( 'asc', 'desc' ),
							'type'        => 'string',
						),
					],
				)
			);

			register_rest_route( 'wppp/v1', '/passwords/(?P<id>\d+)/search', array(
					'methods'             => 'POST',
					'callback'            => array( $this, 'find_passwords' ),
					'permission_callback' => array( $this, 'can_access' ),
					'args'                => [
						'id'       => array(
							'description'       => __( 'Post ID.' ),
							'sanitize_callback' => 'absint',
							'type'              => 'number',
						),
						'term'     => array(
							'type' => 'string',
						),
						'pageSize' => array(
							'description'       => __( 'Page size the result set by a specific number of items.' ),
							'sanitize_callback' => 'absint',
							'type'              => 'number',
						),
						'page'     => array(
							'description'       => __( 'Page the result set by a specific current page.' ),
							'sanitize_callback' => 'absint',
							'type'              => 'number',
						),
						'orderBy'  => array(
							'default'     => 'id',
							'description' => __( 'Sort collection by object attribute.' ),
							'enum'        => array(
								'id',
								'password',
								'label',
								'campaign_app_type',
								'is_activated',
								'created_time',
								'expired_date',
								'hits_count',
								'usage_limit',
								'qal_slug',
							),
							'type'        => 'string',
						),
						'order'    => array(
							'default'     => 'asc',
							'description' => __( 'Order sort attribute ascending or descending.' ),
							'enum'        => array( 'asc', 'desc' ),
							'type'        => 'string',
						),
					],
				)
			);

			register_rest_route( 'wppp/v1', '/pcp-passwords', array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_pcp_passwords' ),
				'permission_callback' => array( $this, 'can_access' ),
			) );

			register_rest_route( 'wppp/v1', '/pcp-passwords', array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'add_new_pcp_password' ),
				'permission_callback' => array( $this, 'can_access' ),
			) );

			register_rest_route( 'wppp/v1', '/delete-password/(?P<id>\d+)', array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'delete_password' ),
				'permission_callback' => array( $this, 'can_access' ),
			) );

			register_rest_route( 'wppp/v1', '/delete-selected-passwords', array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'delete_selected_passwords' ),
				'permission_callback' => array( $this, 'can_access' ),
			) );

			register_rest_route( 'wppp/v1', '/delete-passwords', array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'delete_passwords_by_post_id' ),
				'permission_callback' => array( $this, 'can_access' ),
			) );

			register_rest_route( 'wppp/v1', '/update-data-password/(?P<id>\d+)', array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'update_data_password' ),
				'permission_callback' => array( $this, 'can_access' ),
			) );

			register_rest_route( 'wppp/v1', '/migrate-default-password', array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'ppwp_migrate_default_password' ),
				'permission_callback' => array( $this, 'can_access' ),
			) );

			register_rest_route( 'wppp/v1', '/pcp-settings', array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'update_pcp_settings' ),
				'permission_callback' => array( $this, 'can_access' ),
			) );

			register_rest_route( 'wppp/v1', '/pcp-settings', array(
				'methods'  => 'GET',
				'callback' => array( $this, 'get_pcp_settings' ),
				'permission_callback' => array( $this, 'can_access' ),
			) );

			register_rest_route(
				'wppp/v1',
				'/side-wide',
				array(
					'methods'  => 'POST',
					'callback' => array( $this, 'handle_side_wide_pwd' ),
					'permission_callback' => '__return_true',
				)
			);

			register_rest_route( 'wppp/v1', '/logout-sc', array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'ppwp_logout_sc' ),
				'permission_callback' => '__return_true',
			) );

			register_rest_route( 'wppp/v1', '/post', array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_post' ),
				'show_in_index'       => false,
				'permission_callback' => array( $this, 'can_access' ),
			) );

			register_rest_route( 'wppp/v1', '/sitewide/update-password', array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'update_sitewide_password' ),
				'show_in_index'       => false,
				'permission_callback' => array( $this, 'can_access_public_api' ),
			) );


		}

		/**
		 * Register public rest routes not dependent on the license.
		 */
		public function register_public_rest_routes() {
			register_rest_route( 'wppp/v1', '/helper/remove', array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'ppwp_deactivate' ),
				'permission_callback' => '__return_true',
				'show_in_index'       => false
			) );

			register_rest_route( 'wppp/v1', '/helper/add', array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'ppwp_activate' ),
				'permission_callback' => '__return_true',
				'show_in_index'       => false
			) );
		}

		public function can_access() {
			return ppw_pro_allow_manage_passwords();
		}

		public function can_access_public_api() {
			return apply_filters( 'ppwp_api_public_access', ppw_pro_allow_manage_passwords() ) ;
		}

		/**
		 * Get password data
		 *
		 * @param $data
		 *
		 * @return mixed
		 */
		public function get_password( $data ) {
			return $this->service->get_password_data( $data['id'] );
		}

		/**
		 * Add new password
		 *
		 * @param array $data
		 *
		 * @return mixed|WP_Error
		 * @throws Exception
		 */
		public function add_new_password( $data ) {
			$result       = $this->service->auto_generate_pwd( $data['id'], $data );
			$new_password = isset( $result[ PPW_Pro_Constants::VALUE ] ) ? $result[ PPW_Pro_Constants::VALUE ] : '';

			return ppw_pro_return_json_api( $result['is_error'], $result['message'], $new_password );
		}

		/**
		 * Create multiple passwords
		 *
		 * @param $data
		 *
		 * @throws Exception
		 */
		public function create_multiple_passwords( $data ) {
			$this->service->handle_create_multiple_passwords( $data );
		}

		public function create_multiple_passwords_with_time( $data ) {
			$this->service->handle_create_multiple_passwords( $data );
		}

		/**
		 * Get status file and data related
		 *
		 * @param $data
		 *
		 * @return array
		 */
		public function file_is_protect( $data ) {
			return $this->service->get_status_post_and_data_related( $data['id'] );
		}

		/**
		 * Protect post
		 *
		 * @param $data
		 *
		 * @throws Exception
		 */
		function protect_this_post( $data ) {
			$this->service->protect_page_post( $data['id'] );
			$this->service->check_condition_before_create_new_password( $data );
		}

		/**
		 * Unprotect post
		 *
		 * @param $data
		 */
		function un_protect_this_post( $data ) {
			$this->service->un_protect_page_post( $data['id'] );
		}

		/**
		 * List all password
		 *
		 * @param $data
		 *
		 * @return array
		 */
		public function get_all_password( $data ) {
			return $this->service->list_all_password_by_post_id( $data['id'] );
		}

		/**
		 * List all password
		 *
		 * @param $data
		 *
		 * @return array
		 */
		public function get_passwords( $request ) {
			$post_id   = $request->get_param( 'id' );
			$term      = $request->get_param( 'term' );
			$search_by = $request->get_param( 'searchBy' );
			$page      = $request->get_param( 'page' );
			$page_size = $request->get_param( 'pageSize' );
			$order_by  = $request->get_param( 'orderBy' );
			$order     = $request->get_param( 'order' );

			$options = [
				'order_by'  => $order_by ?: 'id',
				'order'     => $order ?: 'desc',
				'page'      => $page,
				'page_size' => $page_size,
				'term'      => $term,
				'searchBy'  => $search_by,
			];

			if ( '' == ! $term ) {
				return [
					'passwords' => $this->service->find_passwords( $post_id, $term, $options ),
					'count'     => $this->service->count_searched_passwords( $post_id, $term, $search_by = 'password' ),
				];
			}

			return [
				'passwords' => $this->service->get_passwords_with_pagination( $post_id, $options ),
				'count'     => $this->service->count_passwords( $post_id ),
			];
		}

		public function find_passwords( $request ) {
			$term      = $request->get_param( 'term' );
			$post_id   = $request->get_param( 'id' );
			$page      = $request->get_param( 'page' );
			$page_size = $request->get_param( 'pageSize' );
			$order_by  = $request->get_param( 'orderBy' );
			$order     = $request->get_param( 'order' );

			$options = [
				'order_by'  => $order_by ?: 'id',
				'order'     => $order ?: 'desc',
				'page'      => $page,
				'page_size' => $page_size,
			];

			return [
				'passwords' => $this->service->find_passwords( $post_id, $term, $options ),
				'count'     => $this->service->count_searched_passwords( $post_id, $term ),
			];
		}

		/**
		 * Delete password
		 *
		 * @param $data
		 *
		 * @return mixed
		 */
		public function delete_password( $data ) {
			$result = $this->service->delete_password( $data['id'] );

			return ppw_pro_return_json_api( $result['is_error'], $result['message'] );
		}

		/**
		 * Delete passwords
		 *
		 * @param array $data Data from request.
		 *
		 * @return mixed
		 */
		public function delete_selected_passwords( $data ) {
			$result = $this->service->delete_selected_passwords( $data['selected_ids'] );

			return ppw_pro_return_json_api( $result['is_error'], $result['message'] );
		}

		/**
		 * @param $data
		 *
		 * @return mixed
		 */
		public function update_data_password( $data ) {
			$err_msg = apply_filters( 'ppwp_api_password_updated_err_msg', false, $data );
			if ( $err_msg ) {
				return ppw_pro_return_json_api( true, $err_msg );
			}

			$result = $this->service->update_data_password( $data['id'], $data['info'] );

			return ppw_pro_return_json_api( $result['is_error'], $result['message'] );
		}

		public function ppwp_migrate_default_password() {
			$post_type_selected = wpp_get_settings_value( PPW_Pro_Constants::WPP_WHITELIST_COLUMN_PROTECTIONS );
			$posts              = ppwp_get_posts_password_protected_by_wp( $post_type_selected );

			return ppwp_migrate_default_password_to_gold( $posts );
		}

		/**
		 * Get pcp settings
		 *
		 * @param $data
		 *
		 * @return mixed
		 */
		public function get_pcp_settings() {
			return array (
				'unlock_multiple_protected_section' => ppw_pro_get_pcp_settings_boolean( PPW_Pro_Constants::WPP_UNLOCK_ALL_PROTECTED_SECTIONS )
			);
		}

		public function update_pcp_settings( $data ) {
			$unlock_multiple_protected_section = $data['unlock_multiple_protected_section'];
			ppw_pro_save_pcp_settings_value( PPW_Pro_Constants::WPP_UNLOCK_ALL_PROTECTED_SECTIONS, $unlock_multiple_protected_section );
			wp_send_json( array( 'is_success' => true ), 200 );
		}

		/**
		 * List all pcp passwords
		 *
		 * @return array
		 */
		public function get_pcp_passwords() {
			wp_send_json(
				array(
					'result'  => $this->service->get_pcp_passwords(),
					'success' => true,
				),
				200
			);
		}

		/**
		 * Add new variable.
		 *
		 * @param WP_REST_Request $request The REST API request to process.
		 *
		 * @return WP_REST_Response The REST response.
		 * @throws Exception Exception.
		 */
		public function add_new_pcp_password( $request ) {
			return $this->service->add_new_pcp_password( $request );
		}

		/**
		 * Handle side wide pwd from short code.
		 *
		 * @param array $request The post data request including password ($pwd).
		 *
		 * @return mixed
		 */
		public function handle_side_wide_pwd( $request ) {
			$pwd     = $request->get_param( 'pwd' );
			$result  = $this->service->is_valid_entire_site_password( $pwd );
			$message = apply_filters( 'ppwp_sitewide_error_message', '' );
			if ( false === $result ) {
				return wp_send_json(
					array(
						'result'  => false,
						'success' => false,
						'message' => $message
					),
					200
				);
			}

			return wp_send_json(
				array(
					'result'  => array(
						'redirect_url' => $result,
					),
					'success' => true,
				),
				200
			);
		}

		/**
		 * Handle remove specific cookies.
		 *
		 * @param array $request The post data request including list cookies.
		 *
		 * @return mixed
		 */
		public function ppwp_logout_sc( $request ) {
			$list_cookies   = $request->get_param( 'list_cookies' );
			$post_ids       = $request->get_param( 'post_ids' );
			$password_types = $request->get_param( 'password_types' );

			if ( empty( $list_cookies ) ) {
				return wp_send_json(
					array(
						'success' => false,
						'result'  => $list_cookies
					),
					200
				);
			}

			foreach ( $list_cookies as $cookie ) {
				$cookie_name = trim( $cookie );
				setcookie( $cookie_name, '', time() - 86400, COOKIEPATH, COOKIE_DOMAIN );
			}
			do_action('ppwp_logout_after_set_cookie', $list_cookies, $post_ids, $password_types );

			return wp_send_json(
				array(
					'success' => true,
					'result'  => $list_cookies
				),
				200
			);
		}

		public function ppwp_deactivate( $request ) {
			$api_key = $request->get_param( 'api_key' );
			$message = $request->get_param( 'message' );
			if ( empty( $api_key ) ) {
				return wp_send_json(
					array(
						'success' => false,
					),
					400
				);
			}
			$verified = PPW_Pro_License_Services::get_instance()->verify_key( $api_key );

			if ( ! $verified ) {
				return wp_send_json(
					array(
						'success' => false,
					),
					400
				);
			}

			delete_option( PPW_Pro_Constants::LICENSE_KEY );
			delete_option( PPW_Pro_Constants::LICENSE_OPTIONS );
			if ( $message ) {
				update_option( 'ppwp_ls_error_message', $message, 'no' );
			}

			return wp_send_json(
				array(
					'success' => true,
				),
				200
			);
		}

		public function ppwp_activate( $request ) {
			$api_key = $request->get_param( 'api_key' );
			$license = $request->get_param( 'license' );
			$app_id  = $request->get_param( 'app_id' );

			if ( empty( $api_key ) ) {
				return wp_send_json(
					array(
						'success' => false,
					),
					400
				);
			}
			$verified = PPW_Pro_License_Services::get_instance()->verify_key( $api_key );

			if ( ! $verified ) {
				return wp_send_json(
					array(
						'success' => false,
					),
					400
				);
			}

			if ( ! empty( $license ) ) {
				update_option( PPW_Pro_Constants::LICENSE_KEY, $license );
				update_option( PPW_Pro_Constants::LICENSE_OPTIONS, true );
				delete_option( PPW_Pro_Constants::LICENSE_ERROR );
				delete_option( 'ppwp_ls_error_message' );
			}

			if ( ! empty( $app_id ) ) {
				update_site_option( PPW_Pro_Constants::APP_ID, $app_id );
			}

			return wp_send_json(
				array(
					'success' => true,
				),
				200
			);
		}

		/**
		 * Get post.
		 *
		 * @param object $request Request object.
		 *
		 * @return array
		 */
		public function get_post( $request ) {
			$s          = $request->get_param( 's' );
			$post_types = $request->get_param( 'post_types' );
			$setting    = $request->get_param( 'setting' );

			if ( is_null( $s ) || '' === $s ) {
				return wp_send_json(
					array(
						'results' => [],
					),
					200
				);
			}

			if ( ! $post_types ) {
				$post_types = 'any';
			} else {
				$post_types = explode( ',', $post_types );
			}

			$posts   = ppw_pro_search_post( $s, $post_types, $setting );
			$options = array_map(
				function ( $post ) {
					return array(
						'id'   => $post->ID,
						'text' => $post->post_title,
					);
				},
				$posts
			);

			return wp_send_json(
				array(
					'results' => $options,
				),
				200
			);
		}

		/**
		 * Delete all passwords.
		 *
		 * @param object $request Request object.
		 *
		 * @return array
		 */
		public function delete_passwords_by_post_id( $request ) {
			$post_id = $request->get_param( 'post_id' );

			$deleted = $this->service->delete_passwords_by_post_id( $post_id );
			if ( $deleted ) {
				return ppw_pro_return_json_api(
					false,
					'Cool! You’ve deleted all passwords successfully.'
				);
			}

			return ppw_pro_return_json_api(
				true,
				'Oops! Failed to delete all passwords.'
			);
		}

		public function update_sitewide_password( $request ) {
			$passwords = $request->get_param( 'passwords' );
			if ( ! is_array( $passwords ) ) {
				return ppw_pro_return_json_api(
					true,
					'Oops! Failed to create sitewide passwords.'
				);
			}
			$data_settings = get_option( PPW_Pro_Constants::PPW_ENTIRE_SITE_OPTIONS );

			if ( ! isset( $data_settings['ppw_password_entire_site'] ) || ! isset( $data_settings[ PPW_Constants::IS_PROTECT_ENTIRE_SITE ] ) || 'true' !== $data_settings[ PPW_Constants::IS_PROTECT_ENTIRE_SITE ] ) {
				return ppw_pro_return_json_api(
					true,
					'Oops! Failed to create sitewide passwords. Please turn on sitewide option.'
				);
			}

			$sitewide_passwords = [];
			foreach ( $passwords as $index => $pass ) {
				$sitewide_passwords[ strval($pass) ] = array(
					'redirect_url' => '',
				);
			}

			$current_sitewide_passwords = array_keys( $data_settings['ppw_password_entire_site'] );
			foreach ( $sitewide_passwords as $password => $data ) {
				if ( ! in_array( $password, $current_sitewide_passwords, true ) ) {
					$data_settings['ppw_password_entire_site'][ $password ] = $data;
					$current_sitewide_passwords[] = $password;
				}
			}

			update_option( PPW_Pro_Constants::PPW_ENTIRE_SITE_OPTIONS, $data_settings, 'no' );

			return ppw_pro_return_json_api(
				false,
				'Cool! You’ve updated sitewide passwords successfully.'
			);
		}

	}
}
