<?php

namespace GP_Entry_Blocks\Blocks;

use GP_Entry_Blocks\GF_Queryer;

/**
 * Pagination block to page through the current entries. Can be used in conjunction with blocks such as Entries Loop
 * and Entries Table.
 *
 * @since 1.0
 */
class Pagination extends Block {
	public function render( $attributes, $content, $block ) {
		if ( $block->context['gp-entry-blocks/mode'] !== 'multi-entry' ) {
			return '';
		}

		$queryer = GF_Queryer::attach( $block->context );

		if ( ! $queryer || ! $queryer->is_entries() ) {
			return '';
		}

		return $this->pagination_links( $queryer->get_offset(), $queryer->get_pagesize(), $queryer->total_count, $attributes );
	}

	/**
	 * @param $offset number
	 * @param $page_size number
	 * @param $total_count number
	 * @param $attributes array
	 *
	 * @returns string
	 */
	public function pagination_links( $offset, $page_size, $total_count, $attributes ) {
		if ( ! $total_count || $total_count <= $page_size ) {
			return '';
		}

		if ( ! is_numeric( $offset ) ) {
			$offset = 0;
		}

		$pagination_links = '';

		if ( rgar( $attributes, 'showNextPrevious' ) ) {
			if ( $offset !== 0 ) {
				$new_offset = max( 0, $offset - $page_size );

				$previous_label    = rgar( $attributes, 'previousLabel', __( 'Previous', 'gp-entry-blocks' ) );
				$pagination_links .= '<div class="gpeb-pagination-prev"><a href="' . esc_attr( add_query_arg( array( 'offset' => $new_offset ) ) ) . '">&larr; ' . esc_html( $previous_label ) . '</a></div>';
			}
		}

		if ( rgar( $attributes, 'showNumbers' ) ) {
			$pagination_links .= $this->pagination_links_numbers( $offset, $page_size, $total_count, $attributes );
		}

		if ( $offset + $page_size < $total_count && rgar( $attributes, 'showNextPrevious' ) ) {
			$new_offset = $offset + $page_size;

			$next_label        = rgar( $attributes, 'nextLabel', __( 'Next', 'gp-entry-blocks' ) );
			$pagination_links .= '<div class="gpeb-pagination-next"><a href="' . esc_attr( add_query_arg( array( 'offset' => $new_offset ) ) ) . '" class="gpeb-pagination-next">' . esc_html( $next_label ) . ' &rarr;</a></div>';
		}

		return $pagination_links;
	}

	/**
	 * @param $offset number
	 * @param $page_size number
	 * @param $total_count number
	 * @param $attributes array
	 *
	 * @returns string
	 */
	public function pagination_links_numbers( $offset, $page_size, $total_count, $attributes ) {
		if ( ! $total_count || $total_count <= $page_size ) {
			return '';
		}

		if ( ! is_numeric( $offset ) ) {
			$offset = 0;
		}

		$pagination_links = '<div class="gpeb-pagination-numbers">';

		for ( $i = 0; $i < $total_count; $i += $page_size ) {
			// If the current offset matches the GET param, change from a hyperlink to just text.
			if ( rgget( 'offset' ) != $i ) {
				$links[ $i ] = '<a href="' . esc_attr( add_query_arg( array( 'offset' => $i ) ) ) . '">' . ( $i / $page_size + 1 ) . '</a>';
			} else {
				$links[ $i ] = '<span class="gpeb-pagination-current">' . ( $i / $page_size + 1 ) . '</span>';
			}
		}

		/*
		 * Keep links within 2 times the offset in either direction of the current offset.
		 * Always keep the first and last links.
		 */
		$displayed_links = array();

		foreach ( $links as $link_offset => $link ) {
			if ( $link_offset === 0 || $link_offset === $page_size * ( count( $links ) - 1 ) || abs( $link_offset - $offset ) <= 2 * $page_size ) {
				$displayed_links[ $link_offset ] = $link;
			} elseif ( ! isset( $displayed_links['collapsed'] ) ) {
				$displayed_links['collapsed'] = '<span class="gpeb-pagination-numbers-collapsed">...</span>';
			}
		}

		$pagination_links .= implode( ' ', $displayed_links );

		$pagination_links .= '</div>';

		return $pagination_links;
	}
}

